(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 7.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     55872,       1581]
NotebookOptionsPosition[     52154,       1452]
NotebookOutlinePosition[     52825,       1476]
CellTagsIndexPosition[     52782,       1473]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Transmission and Reflection from a Square Well", "Title"],

Cell["Niels Walet ", "Subsubtitle",
 CellChangeTimes->{{3.468841198880681*^9, 3.468841207156838*^9}}],

Cell[CellGroupData[{

Cell["Description", "Section"],

Cell[TextData[{
 "We investigate the reflection and transmission from a square barrier, of \
width ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]]],
 " and height ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "0"], TraditionalForm]]],
 "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Abs", "[", "x", "]"}], "<", "1"}], ",", "1", ",", "0"}], "]"}],
    ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"Hue", "[", "0.0", "]"}]}]}], "]"}]], "Input"],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.], LineBox[CompressedData["
1:eJxN1WlMEwYUAOByjQKjgAoGucYpIDJBBhPnSkQUUK4GpEYpx4QZkHFsSBmG
a4hMWUQExzqEjYIcMhjbFMvRgoJFN8dWwXLJfSlXKS2UcnRbwt7zJS8vX17y
XvL+POOwGEq4PIFASP83/6u2ZJ03MpmMQ9iOmDXT7DUp+udf7SyWxWh7S6+w
qVm0k1Zm/x98tMv40rPv6tCUq8/vHQxGJ3ZlRn/K3gJ3d6u9aQvdBBu0xuxy
0t8A349qV9J4JQWH5NgJ+jLWwJK85GwOWQJmkoZ0C0dXwEWedvEvvhaDMy6G
0q08ROBkherE+HkhODSty+xl5RL4dF/Jx6+YaE/7WOpEMdphQjNHmI8mulOW
SenoWo2ethNn0OvF/TSWClqwWk1vVURP+CbncWUC8J/y+k96xOjSiKD9wlG0
u+3IunUjuqBlsrAoEk2jM1KcwtGWB33O84LRjZUP3ycGoIduXX/yORld4e1S
Q3JGx6qIb1Y5oOVTaUHDVujfD+10/dIMnS/iWmobofdG2ok8d6EFZlN9kyQ0
a5jBSVNBZzB8yvUU0d9WE3WiNxe3LSA3FwzU9E6jR9JqXY/x0FaB/nF6Fej4
/dJiYTL6dtbNAJ+uBXDjsOW7NSbohUSHvWkjc2BTTZej/sWz4F+ssyPHxmfA
wx317zk6ToEnbQrsOQ/GwVopVY62cqNgG8dFm/LlQfBXx+b/YtH5YG3/2twz
u3m4b6VDo9PsGbh8YPKb15lscJCf/m8Dn5SBWarJPY8SmOT//cGYnGE6lQM2
FTStOT96CvZg9j33pvLAbl47FMT3+eAbwfy0Aw2DYK63gbrvjlHwvLF2Co8x
Dk74Yl/oDZ0pMH9QkxLROQMOSfpoWvnULDiPx3o4WTkHfrzP6dpjpQVwRLiy
fnklmljC/ynr5CK4r4lzabUIXd1bQb6wgHYbbKfelqAFi4IPw+QE2yYQWuJS
3ZLeQV9bVqfkqqEDE4poFZpos1XrKLY2mi09kTlnhL5++WWugjmaunX+zh5r
tFAu44G7A9pCpXmmzP2tPtc2stTrrflZP86WUNCnFa8uMM6hr8j8RLmx6Anx
tOxyIbp0bOfugBm0QXOUm1L2Epggla+KHxOCV/1aaB3mIvCAQ7CeeaYYHBuo
qWUyuALWzZ+gKhpIwLVnf6hYpq/h/Y6kune2SMH1tGgT5411sETVVYEVsAmm
/H3Y0aBwC6x6WK2t+Q66raz/5Dkm+gA9KfT7WjTJqCFHtwP99KL9uLYQ7UK0
yiWdkuH+OMmeWj903QC33CsQbVgX0ZQTht4IKJtSTUI3MI2OKN9Ff6a+yL17
D22RyKYcr0fnewZduNKMjl9i3FJ4gbY+G2nI7EWPth+qOjqE9i3sZae/RhPl
Kz2MF9GcqMTuVhH6Us/x4BApevv/g/8BNulpbg==
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{-2, 2}, {0., 1.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.46884079816533*^9}]
}, Open  ]],

Cell[TextData[{
 "The Schrodinger equation in the three regions I: ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "<", 
    RowBox[{"-", "a"}]}], TraditionalForm]]],
 ", II: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"-", "a"}], "<", "x", "<", "a"}], TraditionalForm]]],
 ", and III: ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ">", "a"}], TraditionalForm]]],
 " can easily be written down,\nI, III ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"-", 
      FractionBox[
       SuperscriptBox["\[HBar]", "2"], 
       RowBox[{"2", "m"}]]}], 
     FractionBox[
      SuperscriptBox["d", "2"], 
      SuperscriptBox["dx", "2"]], 
     RowBox[{"\[Phi]", "(", "x", ")"}]}], "=", 
    RowBox[{"E", " ", 
     RowBox[{"\[Phi]", "(", "x", ")"}]}]}], TraditionalForm]]],
 ",\nII:",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        SuperscriptBox["\[HBar]", "2"], 
        RowBox[{"2", "m"}]]}], 
      FractionBox[
       SuperscriptBox["d", "2"], 
       SuperscriptBox["dx", "2"]], 
      RowBox[{"\[Phi]", "(", "x", ")"}]}], "+", 
     RowBox[{
      SubscriptBox["V", "0"], 
      RowBox[{"\[Phi]", "(", "x", ")"}]}]}], "=", 
    RowBox[{"E", " ", 
     RowBox[{"\[Phi]", "(", "x", ")"}]}]}], TraditionalForm]]],
 ".\nWe shall solve these equations for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"0", "<", "E", "<", 
    SubscriptBox["V", "0"]}], TraditionalForm]]],
 ". \nArgue that in that case the solution in regions I and III are \
oscillatory (complex exponents), and in region II is exponential (or the sum \
of hyperbolic functions). \nDefine ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", " ", "=", " ", 
    SqrtBox[
     RowBox[{
      FractionBox[
       RowBox[{"2", "m"}], 
       SuperscriptBox["\[HBar]", "2"]], "E"}]]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Kappa]", "="}], TraditionalForm]]],
 Cell[BoxData[
  FormBox[
   SqrtBox[
    RowBox[{
     FractionBox[
      RowBox[{"2", "m"}], 
      SuperscriptBox["\[HBar]", "2"]], 
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["V", "0"], "-", "E"}], ")"}]}]], TraditionalForm]]],
 ". Is this a sensible definition?"
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "Solving for ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox["T", TraditionalForm]]],
 "."
}], "Section"],

Cell[TextData[{
 "Just to be sure, make ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", ",", "\[Kappa]", ",", 
    SubscriptBox["A", "1"], ",", 
    SubscriptBox["B", "1"], ",", 
    SubscriptBox["B", "2"], ",", 
    SubscriptBox["A", "3"]}], TraditionalForm]]],
 "undefined. When they are not defined this gives some error messages..."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=."}], ";", 
  RowBox[{"\[Kappa]", "=."}], ";", 
  RowBox[{
   SubscriptBox["A", "1"], "=."}], ";", 
  RowBox[{
   SubscriptBox["B", "1"], "=."}], ";", 
  RowBox[{
   SubscriptBox["B", "2"], "=."}], ";", 
  RowBox[{
   SubscriptBox["A", "3"], "=."}], ";"}]], "Input",
 AspectRatioFixed->True],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], 
  RowBox[{
  ":", " "}], "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Assignment on \\\\\\\"\\\", \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\!\\(Subscript\\), \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" for \\\\\\\"\\\", \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\!\\(A\\_1\\), \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" not \
found.\\\\\\\"\\\", \\\"MT\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", \
ButtonNote -> \\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellChangeTimes->{3.468840821936523*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], 
  RowBox[{
  ":", " "}], "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Assignment on \\\\\\\"\\\", \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\!\\(Subscript\\), \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" for \\\\\\\"\\\", \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\!\\(B\\_1\\), \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" not \
found.\\\\\\\"\\\", \\\"MT\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", \
ButtonNote -> \\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellChangeTimes->{3.468840822163591*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"Unset", "::", "\<\"norep\"\>"}], 
  RowBox[{
  ":", " "}], "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Assignment on \\\\\\\"\\\", \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\!\\(Subscript\\), \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" for \\\\\\\"\\\", \\\"MT\\\"]\\)\
\[NoBreak]\\!\\(\\*StyleBox[\\!\\(B\\_2\\), \
\\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" not \
found.\\\\\\\"\\\", \\\"MT\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Unset/norep\\\", \
ButtonNote -> \\\"Unset::norep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellChangeTimes->{3.468840822386681*^9}],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"stop\"\>"}], 
  RowBox[{
  ":", " "}], "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Further output of \\\\\\\"\\\
\", \\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\!\\(Unset :: \
\\\"norep\\\"\\), \\\"MT\\\"]\\)\[NoBreak]\\!\\(\\*StyleBox[\\\"\\\\\\\" will \
be suppressed during this calculation.\\\\\\\"\\\", \\\"MT\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/General/stop\\\", \
ButtonNote -> \\\"General::stop\\\"]\\)\"\>"}]], "Message", "MSG",
 CellChangeTimes->{3.468840822575995*^9}]
}, Open  ]],

Cell["\<\
Define the wave functions in regions I, II and III. We have an incoming and \
reflected wave  in region I, the full solution in region II, and only a \
transmitted wave in region III\
\>", "Text"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    SubscriptBox["\[Phi]", "\[ImaginaryI]"], "[", "x_", "]"}], ":=", 
   RowBox[{
    RowBox[{
     SubscriptBox["A", "1"], " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"\[ImaginaryI]", " ", "k", " ", "x"}]]}], "+", 
    RowBox[{
     SubscriptBox["B", "1"], " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{
       RowBox[{"-", "\[ImaginaryI]"}], " ", "k", " ", "x"}]]}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    SubscriptBox["\[Phi]", "II"], "[", "x_", "]"}], ":=", 
   RowBox[{
    RowBox[{
     SubscriptBox["A", "2"], " ", 
     RowBox[{"Cosh", " ", "[", 
      RowBox[{"\[Kappa]", " ", "x"}], "]"}]}], "+", 
    RowBox[{
     SubscriptBox["B", "2"], " ", 
     RowBox[{"Sinh", " ", "[", 
      RowBox[{"\[Kappa]", " ", "x"}], "]"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    SubscriptBox["\[Phi]", "III"], "[", "x_", "]"}], ":=", 
   RowBox[{
    SubscriptBox["A", "3"], " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[ImaginaryI]", " ", "k", " ", "x"}]]}]}], ";"}]}], "Input",
 CellChangeTimes->{{3.468840839392827*^9, 3.468840858414105*^9}},
 AspectRatioFixed->True],

Cell[TextData[{
 "Set up the four matching equations, for the functions and their derivatives \
at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", 
    RowBox[{"\[PlusMinus]", 
     RowBox[{"a", "."}]}]}], TraditionalForm]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqns1", "=", 
   RowBox[{
    RowBox[{
     SubscriptBox["\[Phi]", "\[ImaginaryI]"], "[", 
     RowBox[{"-", "a"}], "]"}], "\[Equal]", 
    RowBox[{
     SubscriptBox["\[Phi]", "II"], "[", 
     RowBox[{"-", "a"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns2", "=", 
   RowBox[{
    RowBox[{
     SubscriptBox["\[Phi]", "II"], "[", "a", "]"}], "\[Equal]", 
    RowBox[{
     SubscriptBox["\[Phi]", "III"], "[", "a", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns3", "=", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       SubscriptBox["\[Phi]", "\[ImaginaryI]"], ")"}], "\[Prime]",
      MultilineFunction->None], "[", 
     RowBox[{"-", "a"}], "]"}], "\[Equal]", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       SubscriptBox["\[Phi]", "II"], ")"}], "\[Prime]",
      MultilineFunction->None], "[", 
     RowBox[{"-", "a"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns4", "=", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       SubscriptBox["\[Phi]", "II"], ")"}], "\[Prime]",
      MultilineFunction->None], "[", "a", "]"}], "\[Equal]", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       SubscriptBox["\[Phi]", "III"], ")"}], "\[Prime]",
      MultilineFunction->None], "[", "a", "]"}]}]}], ";"}], "\n", 
 RowBox[{"eqns", "=", 
  RowBox[{"{", 
   RowBox[{"eqns1", ",", "eqns2", ",", "eqns3", ",", "eqns4"}], 
   "}"}]}]}], "Input",
 AspectRatioFixed->True],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{"-", "\[ImaginaryI]"}], " ", "a", " ", "k"}]], " ", 
      SubscriptBox["A", "1"]}], "+", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"\[ImaginaryI]", " ", "a", " ", "k"}]], " ", 
      SubscriptBox["B", "1"]}]}], "\[Equal]", 
    RowBox[{
     RowBox[{
      RowBox[{"Cosh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["A", "2"]}], "-", 
     RowBox[{
      RowBox[{"Sinh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["B", "2"]}]}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"Cosh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["A", "2"]}], "+", 
     RowBox[{
      RowBox[{"Sinh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["B", "2"]}]}], "\[Equal]", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"\[ImaginaryI]", " ", "a", " ", "k"}]], " ", 
     SubscriptBox["A", "3"]}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"\[ImaginaryI]", " ", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{"-", "\[ImaginaryI]"}], " ", "a", " ", "k"}]], " ", "k", " ", 
      
      SubscriptBox["A", "1"]}], "-", 
     RowBox[{"\[ImaginaryI]", " ", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"\[ImaginaryI]", " ", "a", " ", "k"}]], " ", "k", " ", 
      SubscriptBox["B", "1"]}]}], "\[Equal]", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "\[Kappa]"}], " ", 
      RowBox[{"Sinh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["A", "2"]}], "+", 
     RowBox[{"\[Kappa]", " ", 
      RowBox[{"Cosh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["B", "2"]}]}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Kappa]", " ", 
      RowBox[{"Sinh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["A", "2"]}], "+", 
     RowBox[{"\[Kappa]", " ", 
      RowBox[{"Cosh", "[", 
       RowBox[{"a", " ", "\[Kappa]"}], "]"}], " ", 
      SubscriptBox["B", "2"]}]}], "\[Equal]", 
    RowBox[{"\[ImaginaryI]", " ", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"\[ImaginaryI]", " ", "a", " ", "k"}]], " ", "k", " ", 
     SubscriptBox["A", "3"]}]}]}], "}"}]], "Output",
 CellChangeTimes->{3.468840868686246*^9}]
}, Open  ]],

Cell[TextData[{
 "Solve the equations. Can you derive the solution yourself? Why don't I \
solve for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["A", "1"], TraditionalForm]]],
 "?"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"solution", "=", 
  RowBox[{"Simplify", "[", 
   RowBox[{"Solve", "[", 
    RowBox[{"eqns", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["A", "3"], ",", 
       SubscriptBox["B", "1"], ",", 
       SubscriptBox["A", "2"], ",", 
       SubscriptBox["B", "2"]}], "}"}]}], "]"}], "]"}]}]], "Input",
 AspectRatioFixed->True],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox["A", "3"], "\[Rule]", 
     FractionBox[
      RowBox[{"2", " ", "\[ImaginaryI]", " ", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{"-", "2"}], " ", "\[ImaginaryI]", " ", "a", " ", "k"}]], " ",
        "k", " ", "\[Kappa]", " ", 
       SubscriptBox["A", "1"]}], 
      RowBox[{
       RowBox[{"2", " ", "\[ImaginaryI]", " ", "k", " ", "\[Kappa]", " ", 
        RowBox[{"Cosh", "[", 
         RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["k", "2"], "-", 
          SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
        RowBox[{"Sinh", "[", 
         RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}]}]]}], ",", 
    RowBox[{
     SubscriptBox["B", "1"], "\[Rule]", 
     FractionBox[
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{"-", "2"}], " ", "\[ImaginaryI]", " ", "a", " ", "k"}]], " ", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["k", "2"], "+", 
         SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
       RowBox[{"Sinh", "[", 
        RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}], " ", 
       SubscriptBox["A", "1"]}], 
      RowBox[{
       RowBox[{"2", " ", "\[ImaginaryI]", " ", "k", " ", "\[Kappa]", " ", 
        RowBox[{"Cosh", "[", 
         RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["k", "2"], "-", 
          SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
        RowBox[{"Sinh", "[", 
         RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}]}]]}], ",", 
    RowBox[{
     SubscriptBox["A", "2"], "\[Rule]", 
     FractionBox[
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{"-", "\[ImaginaryI]"}], " ", "a", " ", "k"}]], " ", "k", " ", 
       SubscriptBox["A", "1"]}], 
      RowBox[{
       RowBox[{"k", " ", 
        RowBox[{"Cosh", "[", 
         RowBox[{"a", " ", "\[Kappa]"}], "]"}]}], "+", 
       RowBox[{"\[ImaginaryI]", " ", "\[Kappa]", " ", 
        RowBox[{"Sinh", "[", 
         RowBox[{"a", " ", "\[Kappa]"}], "]"}]}]}]]}], ",", 
    RowBox[{
     SubscriptBox["B", "2"], "\[Rule]", 
     RowBox[{"-", 
      FractionBox[
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "\[ImaginaryI]"}], " ", "a", " ", "k"}]], " ", "k", 
        " ", 
        SubscriptBox["A", "1"]}], 
       RowBox[{
        RowBox[{"\[ImaginaryI]", " ", "\[Kappa]", " ", 
         RowBox[{"Cosh", "[", 
          RowBox[{"a", " ", "\[Kappa]"}], "]"}]}], "+", 
        RowBox[{"k", " ", 
         RowBox[{"Sinh", "[", 
          RowBox[{"a", " ", "\[Kappa]"}], "]"}]}]}]]}]}]}], "}"}], 
  "}"}]], "Output",
 CellChangeTimes->{3.468840880798787*^9}]
}, Open  ]],

Cell["The next statement removes one of the pair of curly brackets", "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{"substitution", "=", 
   RowBox[{"Flatten", "[", "solution", "]"}]}], ";"}]], "Input",
 AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Transmission Coefficient", "Subsection"],

Cell["\<\
And we now calculate the complex quantity that enters the transmission \
coefficient,\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sqT", "=", 
  RowBox[{
   FractionBox[
    SubscriptBox["A", "3"], 
    SubscriptBox["A", "1"]], "/.", "substitution"}]}]], "Input",
 AspectRatioFixed->True],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", "\[ImaginaryI]", " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "2"}], " ", "\[ImaginaryI]", " ", "a", " ", "k"}]], " ", 
   "k", " ", "\[Kappa]"}], 
  RowBox[{
   RowBox[{"2", " ", "\[ImaginaryI]", " ", "k", " ", "\[Kappa]", " ", 
    RowBox[{"Cosh", "[", 
     RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox["k", "2"], "-", 
      SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
    RowBox[{"Sinh", "[", 
     RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}]}]]], "Output",
 CellChangeTimes->{3.46884089257152*^9}]
}, Open  ]],

Cell[TextData[{
 "Unfortunately, the easiest way to get a maneagable expression for ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]]],
 "and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", " "}], TraditionalForm]]],
 " is to take the numerator and denominator, and multiply those with their \
complex conjugates. ComplexExpand than assumes that all variables are real, \
and Simplify generates a simpler version of the result."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TransmissionCoefficient", "=", 
  RowBox[{"Simplify", "[", 
   FractionBox[
    RowBox[{"ComplexExpand", "[", 
     RowBox[{
      RowBox[{"Numerator", "[", "sqT", "]"}], " ", 
      RowBox[{"Conjugate", "[", 
       RowBox[{"Numerator", "[", "sqT", "]"}], "]"}]}], "]"}], 
    RowBox[{"ComplexExpand", "[", 
     RowBox[{
      RowBox[{"Denominator", "[", "sqT", "]"}], " ", 
      RowBox[{"Conjugate", "[", 
       RowBox[{"Denominator", "[", "sqT", "]"}], "]"}]}], "]"}]], 
   "]"}]}]], "Input"],

Cell[BoxData[
 FractionBox[
  RowBox[{"4", " ", 
   SuperscriptBox["k", "2"], " ", 
   SuperscriptBox["\[Kappa]", "2"]}], 
  RowBox[{
   RowBox[{"4", " ", 
    SuperscriptBox["k", "2"], " ", 
    SuperscriptBox["\[Kappa]", "2"], " ", 
    SuperscriptBox[
     RowBox[{"Cosh", "[", 
      RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}], "2"]}], "+", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["k", "2"], "-", 
       SuperscriptBox["\[Kappa]", "2"]}], ")"}], "2"], " ", 
    SuperscriptBox[
     RowBox[{"Sinh", "[", 
      RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}], "2"]}]}]]], "Output",
 CellChangeTimes->{3.468840914247745*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reflection Coefficient", "Subsection"],

Cell["\<\
We now calculate the complex quantity that enters the reflection coefficient,\
\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sqR", "=", 
  RowBox[{
   FractionBox[
    SubscriptBox["B", "1"], 
    SubscriptBox["A", "1"]], "/.", "\[InvisibleSpace]", 
   "substitution"}]}]], "Input",
 AspectRatioFixed->True],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "2"}], " ", "\[ImaginaryI]", " ", "a", " ", "k"}]], " ", 
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox["k", "2"], "+", 
     SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
   RowBox[{"Sinh", "[", 
    RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}], 
  RowBox[{
   RowBox[{"2", " ", "\[ImaginaryI]", " ", "k", " ", "\[Kappa]", " ", 
    RowBox[{"Cosh", "[", 
     RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox["k", "2"], "-", 
      SuperscriptBox["\[Kappa]", "2"]}], ")"}], " ", 
    RowBox[{"Sinh", "[", 
     RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}]}]}]]], "Output",
 CellChangeTimes->{3.4688409370601482`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ReflectionCoefficient", "=", 
  RowBox[{"Simplify", "[", 
   FractionBox[
    RowBox[{"ComplexExpand", "[", 
     RowBox[{
      RowBox[{"Numerator", "[", "sqR", "]"}], " ", 
      RowBox[{"Conjugate", "[", 
       RowBox[{"Numerator", "[", "sqR", "]"}], "]"}]}], "]"}], 
    RowBox[{"ComplexExpand", "[", 
     RowBox[{
      RowBox[{"Denominator", "[", "sqR", "]"}], " ", 
      RowBox[{"Conjugate", "[", 
       RowBox[{"Denominator", "[", "sqR", "]"}], "]"}]}], "]"}]], 
   "]"}]}]], "Input"],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"2", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["k", "2"], "+", 
       SuperscriptBox["\[Kappa]", "2"]}], ")"}], "2"], " ", 
    SuperscriptBox[
     RowBox[{"Sinh", "[", 
      RowBox[{"2", " ", "a", " ", "\[Kappa]"}], "]"}], "2"]}], 
   RowBox[{
    SuperscriptBox["k", "4"], "-", 
    RowBox[{"6", " ", 
     SuperscriptBox["k", "2"], " ", 
     SuperscriptBox["\[Kappa]", "2"]}], "+", 
    SuperscriptBox["\[Kappa]", "4"], "-", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox["k", "2"], "+", 
        SuperscriptBox["\[Kappa]", "2"]}], ")"}], "2"], " ", 
     RowBox[{"Cosh", "[", 
      RowBox[{"4", " ", "a", " ", "\[Kappa]"}], "]"}]}]}]]}]], "Output",
 CellChangeTimes->{3.4688409459410677`*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Check", "Subsection"],

Cell[TextData[{
 "We now check our calculation, and show that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"R", "+", "T"}], "=", "1."}], TraditionalForm]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"TransmissionCoefficient", "+", "ReflectionCoefficient"}], 
  "]"}]], "Input"],

Cell[BoxData["1"], "Output",
 CellChangeTimes->{3.468840957437855*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plots", "Section"],

Cell[TextData[{
 "We now shall make some plots. Before doing so we realise that ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox["T", TraditionalForm]]],
 " depend on ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", " ", "a"}], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Kappa]", " ", "a"}], TraditionalForm]]],
 " only. Furthermore we have ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[Kappa]", " ", "a"}], " ", "=", "  ", 
    SqrtBox[
     RowBox[{
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          SubscriptBox["\[Kappa]", "0"], "a"}], ")"}], "2"], "-", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"k", " ", "a"}], ")"}], "2"]}], " "}]]}], 
   TraditionalForm]]],
 ", with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], "="}], TraditionalForm]]],
 Cell[BoxData[
  FormBox[
   SqrtBox[
    RowBox[{
     FractionBox[
      RowBox[{"2", "m"}], 
      SuperscriptBox["\[HBar]", "2"]], 
     SubscriptBox["V", "0"]}]], TraditionalForm]]],
 ". We shall investigate ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox["T", TraditionalForm]]],
 " as function of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", " ", "a"}], TraditionalForm]]],
 "."
}], "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{"Rka", "=", 
   RowBox[{"ReflectionCoefficient", "/.", "\[InvisibleSpace]", 
    RowBox[{"\[Kappa]", "\[Rule]", 
     FractionBox[
      SqrtBox[
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[Kappa]", "0"], " ", "a"}], ")"}], "2"], "-", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"k", " ", "a"}], ")"}], "2"]}]], "a"]}]}]}], ";", 
  RowBox[{"Tka", "=", 
   RowBox[{"TransmissionCoefficient", "/.", "\[InvisibleSpace]", 
    RowBox[{"\[Kappa]", "\[Rule]", 
     FractionBox[
      SqrtBox[
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[Kappa]", "0"], " ", "a"}], ")"}], "2"], "-", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"k", " ", "a"}], ")"}], "2"]}]], "a"]}]}]}], ";"}]], "Input",\

 CellChangeTimes->{3.46884096725942*^9}],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], "a"}], "=", "1"}], 
  TraditionalForm]]]], "Subsection"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rka", "/.", "\[InvisibleSpace]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "\[Rule]", "1"}], ",", 
        RowBox[{
         SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}], ",", 
     RowBox[{"Tka", "/.", "\[InvisibleSpace]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "\[Rule]", "1"}], ",", 
        RowBox[{
         SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Hue", "[", "0.0", "]"}], ",", 
      RowBox[{"Hue", "[", "0.6", "]"}]}], "}"}]}], ",", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<k\>\"", ",", "\"\<R or T\>\""}], "}"}]}]}], "]"}]], "Input",\

 CellChangeTimes->{{3.4688410003894587`*^9, 3.46884100597575*^9}, {
  3.468841124625165*^9, 3.468841139222702*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.], LineBox[CompressedData["
1:eJwd1nk0VesbB3BDhsoQSZeSxFGEpAGlvkpFNFxDSjKFDDcVKc3FPVSGUCcJ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     "]]}, 
   {Hue[0.6], LineBox[CompressedData["
1:eJwVk2k8lIsCh6nuMVqItJC1UEoThYqaf5FdxSCEyVpCxlqpENEiSppDixAj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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"k\"", TraditionalForm], 
    FormBox["\"R or T\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{{0, 1}, {0., 0.9999999999999999}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.468840982045105*^9, 3.468841139798126*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], "a"}], "=", "2"}], 
  TraditionalForm]]]], "Subsection"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Rka", "/.", "\[InvisibleSpace]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "\[Rule]", "2"}], ",", 
        RowBox[{
         SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}], ",", 
     RowBox[{"Tka", "/.", "\[InvisibleSpace]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"a", "\[Rule]", "2"}], ",", 
        RowBox[{
         SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Hue", "[", "0.0", "]"}], ",", 
      RowBox[{"Hue", "[", "0.6", "]"}]}], "}"}]}], ",", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<k\>\"", ",", "\"\<R or T\>\""}], "}"}]}]}], "]"}]], "Input",\

 CellChangeTimes->{3.468841149441204*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.], LineBox[CompressedData["
1:eJwd13c4lv37B3AaRkrJipQUoseqr6hU76f0RKEeo6IhojRkNlAqIhJSj1GZ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     "]]}, 
   {Hue[0.6], LineBox[CompressedData["
1:eJwVlmc41Y8DxY1C1DWyC5FUSFo0v6eolFEZlX6hiESlUoQilL3rkr0pZK9C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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"k\"", TraditionalForm], 
    FormBox["\"R or T\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{{0, 1}, {0., 1.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.468840991619035*^9, 3.468841150776752*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], "a"}], "=", 
   RowBox[{"1", "/", "2"}]}], TraditionalForm]]]], "Subsection"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Manipulate", "[", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Rka", "/.", "\[InvisibleSpace]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"a", "\[Rule]", "\[Kappa]a"}], ",", 
          RowBox[{
           SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}], ",", 
       RowBox[{"Tka", "/.", "\[InvisibleSpace]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"a", "\[Rule]", "\[Kappa]a"}], ",", 
          RowBox[{
           SubscriptBox["\[Kappa]", "0"], "\[Rule]", "1"}]}], "}"}]}]}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Hue", "[", "0.0", "]"}], ",", 
        RowBox[{"Hue", "[", "0.6", "]"}]}], "}"}]}], ",", 
     RowBox[{"AxesLabel", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<k\>\"", ",", "\"\<R or T\>\""}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Kappa]a", ",", "0.1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.468841034112748*^9, 3.468841101022643*^9}, {
  3.4688411575625553`*^9, 3.468841164807399*^9}}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`\[Kappa]a$$ = 0.706, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{
      Hold[$CellContext`\[Kappa]a$$], 0.1, 2}}, Typeset`size$$ = {
    360., {118., 122.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`\[Kappa]a$7382$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`\[Kappa]a$$ = 0.1}, 
      "ControllerVariables" :> {
        Hold[$CellContext`\[Kappa]a$$, $CellContext`\[Kappa]a$7382$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> Plot[{
         ReplaceAll[$CellContext`Rka, {$CellContext`a -> \
$CellContext`\[Kappa]a$$, Subscript[$CellContext`\[Kappa], 0] -> 1}], 
         ReplaceAll[$CellContext`Tka, {$CellContext`a -> \
$CellContext`\[Kappa]a$$, Subscript[$CellContext`\[Kappa], 0] -> 
           1}]}, {$CellContext`k, 0, 1}, PlotStyle -> {
          Hue[0.], 
          Hue[0.6]}, AxesLabel -> {"k", "R or T"}], 
      "Specifications" :> {{$CellContext`\[Kappa]a$$, 0.1, 2}}, 
      "Options" :> {}, "DefaultOptions" :> {}],
     ImageSizeCache->{405., {162., 167.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{{3.4688410929980497`*^9, 3.468841102009666*^9}, 
   3.468841165391594*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Assignments", "Section"],

Cell[CellGroupData[{

Cell[TextData[{
 "Solve for ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox["T", TraditionalForm]]],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", ">", 
    SubscriptBox["V", "0"]}], TraditionalForm]]],
 "(Following the schema above). Provide graphs for the same three values of \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], 
    RowBox[{"a", "."}]}], TraditionalForm]]]
}], "Subsection"],

Cell[TextData[{
 "Hints: What is the wave function in region II? What are allowed values of \
",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{"k", " ", "a"}]}], TraditionalForm]]],
 "?"
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "Change the barrier into a well, i.e., ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["V", "0"], "\[Rule]", 
    RowBox[{"-", 
     SubscriptBox["V", "0"]}]}], TraditionalForm]]],
 "(Following the schema above). Provide graphs for the same three values of \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Kappa]", "0"], 
    RowBox[{"a", "."}]}], TraditionalForm]]]
}], "Subsection"],

Cell["\<\
Hints: What is the wave function in region II? What are allowed values of  k \
a?\
\>", "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
WindowToolbars->{},
CellGrouping->Automatic,
WindowSize->{1101, 948},
WindowMargins->{{Automatic, 138}, {Automatic, 3}},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, 128}},
ShowCellLabel->True,
ShowCellTags->False,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
CharacterEncoding->"XAutomaticEncoding",
FrontEndVersion->"7.0 for Linux x86 (64-bit) (February 25, 2009)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "iso8859-1"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[567, 22, 63, 0, 75, "Title"],
Cell[633, 24, 101, 1, 31, "Subsubtitle"],
Cell[CellGroupData[{
Cell[759, 29, 30, 0, 78, "Section"],
Cell[792, 31, 258, 10, 32, "Text"],
Cell[CellGroupData[{
Cell[1075, 45, 371, 12, 42, "Input"],
Cell[1449, 59, 1803, 36, 263, "Output"]
}, Open  ]],
Cell[3267, 98, 2215, 80, 207, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[5519, 183, 162, 8, 78, "Section"],
Cell[5684, 193, 345, 10, 32, "Text"],
Cell[CellGroupData[{
Cell[6054, 207, 332, 12, 42, "Input"],
Cell[6389, 221, 710, 13, 24, "Message"],
Cell[7102, 236, 710, 13, 24, "Message"],
Cell[7815, 251, 710, 13, 24, "Message"],
Cell[8528, 266, 628, 11, 24, "Message"]
}, Open  ]],
Cell[9171, 280, 206, 4, 53, "Text"],
Cell[9380, 286, 1193, 38, 87, "Input"],
Cell[10576, 326, 238, 8, 32, "Text"],
Cell[CellGroupData[{
Cell[10839, 338, 1482, 49, 131, "Input"],
Cell[12324, 389, 2482, 73, 66, "Output"]
}, Open  ]],
Cell[14821, 465, 193, 7, 32, "Text"],
Cell[CellGroupData[{
Cell[15039, 476, 358, 11, 42, "Input"],
Cell[15400, 489, 2932, 82, 117, "Output"]
}, Open  ]],
Cell[18347, 574, 76, 0, 32, "Text"],
Cell[18426, 576, 145, 4, 42, "Input"],
Cell[CellGroupData[{
Cell[18596, 584, 46, 0, 39, "Subsection"],
Cell[18645, 586, 109, 3, 32, "Text"],
Cell[CellGroupData[{
Cell[18779, 593, 181, 6, 66, "Input"],
Cell[18963, 601, 668, 18, 66, "Output"]
}, Open  ]],
Cell[19646, 622, 438, 11, 53, "Text"],
Cell[CellGroupData[{
Cell[20109, 637, 522, 14, 67, "Input"],
Cell[20634, 653, 682, 21, 67, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21365, 680, 44, 0, 39, "Subsection"],
Cell[21412, 682, 103, 3, 32, "Text"],
Cell[CellGroupData[{
Cell[21540, 689, 206, 7, 66, "Input"],
Cell[21749, 698, 812, 23, 68, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[22598, 726, 520, 14, 67, "Input"],
Cell[23121, 742, 843, 26, 70, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[24013, 774, 27, 0, 39, "Subsection"],
Cell[24043, 776, 175, 6, 32, "Text"],
Cell[CellGroupData[{
Cell[24243, 786, 129, 3, 42, "Input"],
Cell[24375, 791, 70, 1, 41, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[24506, 799, 24, 0, 78, "Section"],
Cell[24533, 801, 1346, 55, 86, "Text"],
Cell[25882, 858, 933, 29, 76, "Input"],
Cell[CellGroupData[{
Cell[26840, 891, 152, 5, 39, "Subsection"],
Cell[CellGroupData[{
Cell[27017, 900, 1023, 30, 65, "Input"],
Cell[28043, 932, 7294, 129, 265, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[35386, 1067, 152, 5, 39, "Subsection"],
Cell[CellGroupData[{
Cell[35563, 1076, 949, 29, 65, "Input"],
Cell[36515, 1107, 10803, 186, 265, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[47367, 1299, 173, 5, 39, "Subsection"],
Cell[CellGroupData[{
Cell[47565, 1308, 1221, 33, 65, "Input"],
Cell[48789, 1343, 1963, 39, 356, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[50813, 1389, 30, 0, 78, "Section"],
Cell[CellGroupData[{
Cell[50868, 1393, 466, 19, 39, "Subsection"],
Cell[51337, 1414, 207, 8, 32, "Text"]
}, Open  ]],
Cell[CellGroupData[{
Cell[51581, 1427, 425, 15, 59, "Subsection"],
Cell[52009, 1444, 105, 3, 32, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
