(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 7.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     69544,       1369]
NotebookOptionsPosition[     67768,       1305]
NotebookOutlinePosition[     68181,       1321]
CellTagsIndexPosition[     68138,       1318]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Quantum-Classical correspondence", "Title"],

Cell[TextData[{
 "In this ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " notebook everything is expressed in terms of the harmonic oscillator \
length parameter ",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", "=", 
    SqrtBox[
     FormBox[
      FractionBox["\[HBar]", 
       RowBox[{"m", " ", "\[Omega]"}]],
      TraditionalForm]]}], TraditionalForm]]],
 ". "
}], "Text"],

Cell[CellGroupData[{

Cell["Some useful definitions", "Section"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"A", "(", "n", ")"}], TraditionalForm]]],
 " is the amplitude of classical motion for given ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"A", "[", "n_", "]"}], "=", 
  RowBox[{"b", " ", 
   SqrtBox[
    RowBox[{
     RowBox[{"2", " ", "n"}], "+", "1"}]]}]}]], "Input",
 CellChangeTimes->{{3.468840455638266*^9, 3.468840458238749*^9}}],

Cell[BoxData[
 RowBox[{"b", " ", 
  SqrtBox[
   RowBox[{"1", "+", 
    RowBox[{"2", " ", "n"}]}]]}]], "Output",
 CellChangeTimes->{3.468840480392974*^9}]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Phi]", "n"], TraditionalForm]]],
 "is the normalised wave function of the harmonic oscillator"
}], "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["\[Phi]", "n_"], "[", "x_", "]"}], ":=", 
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"-", 
     FractionBox["n", "2"]}]], " ", 
   FractionBox["1", 
    SqrtBox[
     RowBox[{"n", "!"}]]], " ", 
   FractionBox["1", 
    SqrtBox["b"]], " ", 
   SuperscriptBox["\[Pi]", 
    RowBox[{
     RowBox[{"-", "1"}], "/", "4"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], 
      RowBox[{"2", " ", 
       SuperscriptBox["b", "2"]}]]}]], " ", 
   RowBox[{"HermiteH", "[", 
    RowBox[{"n", ",", 
     FractionBox["x", "b"]}], "]"}]}]}]], "Input"],

Cell[TextData[{
 "For example, look at ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Phi]", "0"], TraditionalForm]]],
 ":"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  SubscriptBox["\[Phi]", "0"], "[", "x", "]"}]], "Input"],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], 
     RowBox[{"2", " ", 
      SuperscriptBox["b", "2"]}]]}]], 
  RowBox[{
   SqrtBox["b"], " ", 
   SuperscriptBox["\[Pi]", 
    RowBox[{"1", "/", "4"}]]}]]], "Output",
 CellChangeTimes->{{3.468840477286325*^9, 3.4688404856311817`*^9}}]
}, Open  ]],

Cell[TextData[{
 "I might as well take ",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", "=", "1"}], TraditionalForm]]],
 ", since everything is expressed in it."
}], "Text",
 CellChangeTimes->{{3.4688404913769207`*^9, 3.468840491829405*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", "1"}]], "Input"],

Cell[BoxData["1"], "Output",
 CellChangeTimes->{3.468840500771328*^9}]
}, Open  ]],

Cell[TextData[{
 "Let me now define a function that plots the classical and quantum \
probabilities for given ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 ","
}], "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{"pl", "[", "n_", "]"}], ":=", 
  RowBox[{"Plot", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      SuperscriptBox[
       RowBox[{
        SubscriptBox["\[Phi]", "n"], "[", "x", "]"}], "2"], ",", 
      FractionBox["1", 
       RowBox[{"\[Pi]", " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox[
           RowBox[{"A", "[", "n", "]"}], "2"], "-", 
          SuperscriptBox["x", "2"]}]]}]]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", 
       RowBox[{"A", "[", "n", "]"}]}], ",", 
      RowBox[{"A", "[", "n", "]"}]}], "}"}], ",", 
    RowBox[{"PlotStyle", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Hue", "[", "0.", "]"}], ",", 
       RowBox[{"Hue", "[", "0.6", "]"}]}], "}"}]}]}], "]"}]}]], "Input"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "Investigate this for various values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 "."
}], "Section"],

Cell[TextData[{
 "First for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "0"}], TraditionalForm]]],
 ". Do you understand this behaviour?"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pl", "[", "0", "]"}]], "Input"],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.], LineBox[CompressedData["
1:eJw113k0ldv/B/BMmUuD3DKrDCeJSAl9IldSMqV0M3eja06GXCUZypCSORkj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     "]]}, 
   {Hue[0.6], LineBox[CompressedData["
1:eJw12Xk4Vd/XAHCpFGWMRiQzyUxRWpISRaWSoZBQJEpkSBSSIfOQOULGkJQh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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0.2},
  PlotRange->{{-1, 1}, {0.20755376565346084`, 0.89401567012826}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.46884051751442*^9}]
}, Open  ]],

Cell[TextData[{
 "Then for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "10"}], TraditionalForm]]],
 ". "
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pl", "[", "10", "]"}]], "Input"],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.], LineBox[CompressedData["
1:eJwUV3c81t8Xt/d4jEqDkKRCUVEqRypNhYRE3yIRZY8ie0ZGVhllEyF7c+9j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     "]]}, 
   {Hue[0.6], LineBox[CompressedData["
1:eJw12Xc4le8bAHB7HOucMiIpKklCQmXdL2WUUVmRaJAdQjQopVBGZVeUEBK+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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->
   NCache[{{-21^Rational[1, 2], 21^Rational[1, 2]}, {0., 
     0.2963299233142369}}, {{-4.58257569495584, 4.58257569495584}, {0., 
    0.2963299233142369}}],
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.468840527698456*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Manipulate", "[", 
  RowBox[{
   RowBox[{"pl", "[", "i", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "10", ",", "100", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.468840422621052*^9, 3.468840444544978*^9}}],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`i$$ = 100, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{
      Hold[$CellContext`i$$], 10, 100, 1}}, Typeset`size$$ = {
    360., {116., 119.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`i$1205$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`i$$ = 10}, 
      "ControllerVariables" :> {
        Hold[$CellContext`i$$, $CellContext`i$1205$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> $CellContext`pl[$CellContext`i$$],
       "Specifications" :> {{$CellContext`i$$, 10, 100, 1}}, "Options" :> {}, 
      "DefaultOptions" :> {}],
     ImageSizeCache->{405., {171., 176.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellChangeTimes->{3.46884043683321*^9, 3.468840529754849*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Assignments", "Section"],

Cell[CellGroupData[{

Cell["Can you take a running average of the quantum result?", "Subsection"],

Cell[TextData[{
 "Hint: Define Av[f_,x_] = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"x", "-", "\[Epsilon]"}], 
     RowBox[{"x", "+", "\[Epsilon]"}]], 
    RowBox[{"f", 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]]],
 ", and choose \[Epsilon] the right way"
}], "Text"]
}, Open  ]],

Cell["What conclusion(s) can you draw from these results?", "Subsection"]
}, Open  ]]
}, Open  ]]
},
WindowSize->{1239, 866},
WindowMargins->{{Automatic, 153}, {Automatic, 0}},
FrontEndVersion->"7.0 for Linux x86 (64-bit) (February 25, 2009)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "iso8859-1"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[567, 22, 49, 0, 75, "Title"],
Cell[619, 24, 381, 15, 57, "Text"],
Cell[CellGroupData[{
Cell[1025, 43, 42, 0, 78, "Section"],
Cell[1070, 45, 214, 8, 32, "Text"],
Cell[CellGroupData[{
Cell[1309, 57, 231, 7, 48, "Input"],
Cell[1543, 66, 153, 5, 46, "Output"]
}, Open  ]],
Cell[1711, 74, 167, 5, 32, "Text"],
Cell[1881, 81, 660, 24, 75, "Input"],
Cell[2544, 107, 136, 6, 32, "Text"],
Cell[CellGroupData[{
Cell[2705, 117, 81, 2, 42, "Input"],
Cell[2789, 121, 367, 12, 81, "Output"]
}, Open  ]],
Cell[3171, 136, 237, 7, 32, "Text"],
Cell[CellGroupData[{
Cell[3433, 147, 49, 1, 42, "Input"],
Cell[3485, 150, 70, 1, 41, "Output"]
}, Open  ]],
Cell[3570, 154, 179, 6, 32, "Text"],
Cell[3752, 162, 809, 26, 77, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[4598, 193, 129, 5, 78, "Section"],
Cell[4730, 200, 155, 6, 32, "Text"],
Cell[CellGroupData[{
Cell[4910, 210, 55, 1, 42, "Input"],
Cell[4968, 213, 14733, 249, 264, "Output"]
}, Open  ]],
Cell[19716, 465, 122, 6, 32, "Text"],
Cell[CellGroupData[{
Cell[19863, 475, 56, 1, 42, "Input"],
Cell[19922, 478, 45391, 753, 266, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[65350, 1236, 254, 6, 42, "Input"],
Cell[65607, 1244, 1526, 31, 374, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67182, 1281, 30, 0, 78, "Section"],
Cell[CellGroupData[{
Cell[67237, 1285, 75, 0, 39, "Subsection"],
Cell[67315, 1287, 337, 11, 34, "Text"]
}, Open  ]],
Cell[67667, 1301, 73, 0, 39, "Subsection"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
